<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Intervention\Image\Facades\Image;
use App\Models\Product;

class ProductController extends Controller
{
    public function getAddProduct()
    {
        return view('majik.addproduct');
    }

    public function store(Request $request)
    {
        // Validate the incoming request data
        $request->validate([
            'product_name' => 'required|string|max:255',
            'product_sku' => 'required|string|max:255|unique:products,sku',
            'product_price' => 'required|numeric|min:0',
            'product_count' => 'required|integer|min:0',
            'product_desc' => 'required|string',
            'product_image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $imagePath = null;

        if ($request->hasFile('product_image')) {
            $image = $request->file('product_image');

            $resizedImage = Image::make($image)->resize(250, 250)->encode('jpg');

            $targetDirectory = public_path('assets/web-new/assets/images/products/');
            $fileName = uniqid() . '.jpg';
            $imagePath = 'assets/web-new/assets/images/products/' . $fileName;

            if (!file_exists($targetDirectory)) {
                mkdir($targetDirectory, 0777, true);
            }
            $resizedImage->save($targetDirectory . $fileName);
        }

        Product::create([
            'name' => $request->product_name,
            'sku' => $request->product_sku,
            'price' => $request->product_price,
            'quantity' => $request->product_count,
            'description' => $request->product_desc,
            'image_path' => $imagePath, 
        ]);

        return redirect()->route('productlist')->with('success', 'Product added successfully.');
    }

    public function getProducts()
    {
        $products = Product::all();
        return response()->json($products);

        // return view('majik.productlist', compact('products'));
    }

    public function purchaseProduct($id)
    {
        $product = Product::findOrFail($id);
        return view('majik.purchase', compact('product'));
    }

    public function getProduct($id)
{
    $product = Product::findOrFail($id);
    return response()->json($product);
}

/**
 * Update product data
 */
public function updateProduct(Request $request, $id)
{
    $request->validate([
        'name' => 'required|string|max:255',
        'sku' => 'required|string|max:255|unique:products,sku,'.$id,
        'price' => 'required|numeric|min:0',
        'quantity' => 'required|integer|min:0',
        'description' => 'required|string',
        'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
    ]);

    $product = Product::findOrFail($id);
    $data = $request->only(['name', 'sku', 'price', 'quantity', 'description']);

    // Handle image update if provided
    if ($request->hasFile('image')) {
        $image = $request->file('image');
        $resizedImage = Image::make($image)->resize(250, 250)->encode('jpg');
        
        $targetDirectory = public_path('assets/web-new/assets/images/products/');
        $fileName = uniqid() . '.jpg';
        $imagePath = 'assets/web-new/assets/images/products/' . $fileName;

        if (!file_exists($targetDirectory)) {
            mkdir($targetDirectory, 0777, true);
        }
        $resizedImage->save($targetDirectory . $fileName);
        
        // Delete old image if it exists
        if ($product->image_path && file_exists(public_path($product->image_path))) {
            unlink(public_path($product->image_path));
        }
        
        $data['image_path'] = $imagePath;
    }

    $product->update($data);

    return response()->json([
        'success' => true,
        'message' => 'Product updated successfully'
    ]);
}

/**
 * Delete product
 */
public function deleteProduct($id)
{
    $product = Product::findOrFail($id);
    
    // Delete associated image
    if ($product->image_path && file_exists(public_path($product->image_path))) {
        unlink(public_path($product->image_path));
    }
    
    $product->delete();

    return response()->json([
        'success' => true,
        'message' => 'Product deleted successfully'
    ]);
}
}